<?php

declare(strict_types=1);

namespace App\Http\Livewire;

use App\Models\OnlineMeetingSignup as OnlineMeetingSignupModel;
use App\Schedules\OnlineMeetingSchedule;
use Carbon\Carbon;
use Closure;
use Filament\Forms\Components\Checkbox;
use Filament\Forms\Components\Radio;
use Filament\Forms\Components\TextInput;
use Illuminate\Support\HtmlString;

class OnlineMeetingSignup extends SignupComponent
{
    public ?string $formattedEventDate = null;

    protected Carbon $start_date;

    protected Carbon $end_date;

    public function __construct()
    {
        parent::__construct();

        $this->start_date = OnlineMeetingSchedule::start();

        $this->end_date = OnlineMeetingSchedule::end();
    }

    public function isFormOpen(): bool
    {
        return OnlineMeetingSchedule::isOpen() &&
            (
                ! OnlineMeetingSignupModel::registrationsFullForDate('2024-10-08') ||
                ! OnlineMeetingSignupModel::registrationsFullForDate('2024-10-09')
            );
    }

    protected function getFormSchema(): array
    {
        return [

            TextInput::make('unitate')
                ->label('Denumire completă unitate de învățământ')
                ->required(),

            TextInput::make('city')
                ->label('Localitate')
                ->maxLength(200)
                ->required(),

            TextInput::make('county')
                ->label('Județ')
                ->maxLength(200)
                ->required(),

            Radio::make('statut')
                ->label('Statut')
                ->columns(2)
                ->options([
                    'profesor' => 'Profesor',
                    'elev' => 'Elev',
                ])
                ->default('profesor')
                ->reactive()
                ->required(),

            TextInput::make('nume')
                ->label('Nume și prenume')
                ->required(),

            TextInput::make('telefon')
                ->label('Număr de telefon')
                ->tel()
                ->required(),

            TextInput::make('email')
                ->label('Email')
                ->email()
                ->unique()
                ->required(),

            TextInput::make('numar_elevi')
                ->label('Număr de elevi participanți la întâlnire')
                ->integer()
                ->required(fn (Closure $get) => $get('statut') === 'profesor')
                ->hidden(fn (Closure $get) => $get('statut') !== 'profesor'),

            Radio::make('clasa')
                ->label('Clasa')
                ->columns(4)
                ->options([
                    'IX' => 'a IX-a',
                    'X' => 'a X-a',
                    'XI' => 'a XI-a',
                    'XII' => 'a XII-a',
                ])
                ->required(fn (Closure $get) => $get('statut') === 'elev')
                ->hidden(fn (Closure $get) => $get('statut') !== 'elev'),

            Radio::make('data')
                ->label('Dată participare la întâlnirea online')
                ->columns(2)
                ->options([
                    '2024-10-08' => '8 octombrie 2024',
                    '2024-10-09' => '9 octombrie 2024',
                ])
                ->descriptions(
                    fn (Radio $component) => collect($component->getOptions())
                        ->keys()
                        ->mapWithKeys(fn (string $date) => [
                            $date => OnlineMeetingSignupModel::registrationsFullForDate($date)
                                ? 'Nu mai sunt locuri disponibile'
                                : '10:00 – 12:00',
                        ])
                        ->all()
                )
                ->disableOptionWhen(
                    fn (string $value) => OnlineMeetingSignupModel::registrationsFullForDate($value)
                )
                ->required(),

            Checkbox::make('rules')
                ->label(
                    new HtmlString('Am citit și sunt de acord cu <a href="https://romania.representation.ec.europa.eu/lider-european_ro" class="text-blue-600 hover:underline" target="_blank">regulamentul campaniei</a>.')
                )
                ->required(),

            Checkbox::make('gdpr')
                ->label(
                    new HtmlString('Sunt de acord cu <a href="/gdpr.pdf" class="text-blue-600 hover:underline" target="_blank">prelucrarea datelor personale</a>.')
                )
                ->required(),
        ];
    }

    protected function getFormModel(): string
    {
        return OnlineMeetingSignupModel::class;
    }

    public function submit(): void
    {
        $model = $this->form->model::create(
            $this->form->getState()
        );

        $model->sendNotification();

        $this->markAsRegistered(
            Carbon::parse($this->data['data'])->isoFormat('DD MMMM Y')
        );
    }
}
