<?php

declare(strict_types=1);

namespace App\Http\Livewire;

use App\Models\ContestSignup as ContestSignupModel;
use App\Schedules\ContestSchedule;
use Carbon\Carbon;
use Filament\Forms\Components\Checkbox;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\Fieldset;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Illuminate\Support\HtmlString;

class ContestSignup extends SignupComponent
{
    protected Carbon $start_date;

    protected Carbon $end_date;

    public function __construct()
    {
        parent::__construct();

        $this->start_date = ContestSchedule::start();

        $this->end_date = ContestSchedule::end();
    }

    public function isFormOpen(): bool
    {
        return ContestSchedule::isOpen();
    }

    protected function getFormSchema(): array
    {
        return [

            TextInput::make('unitate')
                ->label('Denumire completă unitate învățământ')
                ->maxLength(200)
                ->required(),

            TextInput::make('city')
                ->label('Localitate')
                ->maxLength(200)
                ->required(),

            TextInput::make('county')
                ->label('Județ')
                ->maxLength(200)
                ->required(),

            TextInput::make('nume')
                ->label('Nume și prenume profesor')
                ->maxLength(200)
                ->required(),

            TextInput::make('telefon')
                ->label('Număr de telefon')
                ->tel()
                ->required(),

            TextInput::make('email')
                ->label('Email')
                ->email()
                ->unique()
                ->required(),

            TextInput::make('nume_echipa')
                ->label('Nume echipă')
                ->maxLength(200)
                ->required(),

            Fieldset::make('Componență echipă')
                ->columns(3)
                ->schema(
                    $this->getTeamSchema(4)
                ),

            Checkbox::make('rules')
                ->label(
                    new HtmlString('Am citit și sunt de acord cu <a href="https://romania.representation.ec.europa.eu/lider-european_ro" class="text-blue-600 hover:underline" target="_blank">regulamentul campaniei</a>.')
                )
                ->required(),

            Checkbox::make('gdpr')
                ->label(
                    new HtmlString('Sunt de acord cu <a href="/gdpr.pdf" class="text-blue-600 hover:underline" target="_blank">prelucrarea datelor personale</a>.')
                )
                ->required(),
        ];
    }

    protected function getTeamSchema(int $count = 1): array
    {
        return collect(range(1, $count))
            ->map(fn (int $id, int $index) => [
                TextInput::make("echipa.{$index}.nume")
                    ->label('Nume și prenume')
                    ->placeholder("Elev $id")
                    ->maxLength(200)
                    ->required(),

                DatePicker::make("echipa.{$index}.data_nasterii")
                    ->label('Data nașterii')
                    ->maxDate(today())
                    ->displayFormat('d.m.Y')
                    ->required(),

                Select::make("echipa.{$index}.clasa")
                    ->label('Clasa')
                    ->options([
                        'IX' => 'a IX-a',
                        'X' => 'a X-a',
                        'XI' => 'a XI-a',
                        'XII' => 'a XII-a',
                    ])
                    ->required(),
            ])
            ->flatten()
            ->all();
    }

    protected function getFormModel(): string
    {
        return ContestSignupModel::class;
    }

    public function submit(): void
    {
        $model = $this->form->model::create(
            $this->form->getState()
        );

        $model->sendNotification();

        $this->markAsRegistered(
            $model->nume_echipa
        );
    }

    public function registeredTeam(): ?string
    {
        return session()->get($this->registeredSessionKey());
    }
}
